
rm(list=ls())
library(dplyr)
library(Synth)

dat = read.csv("RAS.csv", header=TRUE)

dat$country = as.character(dat$country)

dat = dat[, c("country", "ctyid", "year", "polity", "gdppc", "pcturban", "rdi", "muslim", "pctmuslim", 
              "state", "social", "legislation")]


set.seed(12345)

##############################################################
# Create not.in function to make life easier
##############################################################
`%not.in%` <- function(x,y)!('%in%'(x,y))



##############################################################
# Delete Incomplete Countries as Preparation for Synthetic Control
##############################################################

num.missing <- function(x) {
  return (sum(is.na(x)))
}

dat.temp = dat
out = aggregate(x=dat.temp, by=list(country=dat.temp$country), FUN=num.missing)

has.missing = rowSums((out[,-1])) > 0
countries.without.missing = out$country[!has.missing]

dat = dat[dat$country %in% countries.without.missing,]



##############################################################
# defining global graphic parameters
##############################################################
# global parameters for graphic
width = 3200
height = 2400
res = 300

predictors.global = c("polity", "gdppc", "pcturban", "rdi", 
                      "state", "social", "legislation")

treatment.identifier = "Indonesia"

controls.identifier = unique(dat$country[dat$country %not.in% treatment.identifier])

time = 1990:2014
time.pretreat = 1990:1997

# global parameters for graphic
width = 3200
height = 2400
res = 300



##############################################################
# synthetic control estimation
##############################################################

build.data <- function(dat.synth, predictors, dependent, treatment.identifier, controls.identifier, 
                       time, time.pretreat) {
  
  return(
    dataprep(
      foo = dat.synth,
      predictors = predictors,
      predictors.op = "mean",
      dependent = dependent,
      unit.variable = "ctyid",
      time.variable = "year",
      treatment.identifier = treatment.identifier,
      controls.identifier = controls.identifier,
      time.predictors.prior = time.pretreat,
      time.optimize.ssr = time.pretreat,
      unit.names.variable = "country",
      time.plot = time
    )
  )
}


# Draw Trends of Indonesia and Synthetic Indonesia
draw.trends <- function(Y0, Y1, x.range, ylim, ylab, legend.lab, time.v, 
                        text.x, text.y, text) {

    plot(x.range,Y1,
       type="l", ylim=ylim, 
       col=rgb(0,0,0,alpha=.3), lty="solid",
       ylab = ylab,
       xlab ="Year",
       lwd=6,
       axes = "false",
       cex.lab=1.5
  )
  axis(side=1, at=seq(time[1], time[length(time)], 2))
  axis(side=2, at=seq(ylim[1], ylim[2], 5))
  
  lines(x.range,Y0,col="black",lty="dashed",lwd=2)
  abline(v=time.v, lty="dotted")
  text(text.x, text.y, text, cex=1.4)
  legend(x="bottomright",
         legend=legend.lab,
         lty=c("solid","dashed"),col=c("black","black"),
         cex=1.4,bg="white",lwd=c(2,2))
  
}



##############################################################
# state
##############################################################

dependent = "state"
predictors = unique(c(predictors.global, dependent))


dataprep.out <- build.data(dat.synth = dat,
                           predictors = predictors, 
                           dependent = dependent, 
                           treatment.identifier, 
                           controls.identifier,
                           time,
                           time.pretreat)

synth.out <- synth(dataprep.out)


png(filename = "Figure 4. Trajectories of State Discrimination.png", width=width, height=height, res=res)
draw.trends(Y0 = (dataprep.out$Y0%*%synth.out$solution.w),
            Y1 = dataprep.out$Y1plot,
            x.range = time,
            ylim = c(0, 45),
            ylab = "State Discrimination of Religious Minorities",
            legend.lab = c("Indonesia","Synthetic Indonesia"),
            time.v = 1998,
            text.x = 1998, text.y = 10, text = "Soeharto's\nDownfall"
            )
dev.off()

synth.tables <- synth.tab(
  dataprep.res = dataprep.out,
  synth.res = synth.out)
sink(file = paste0("Synthetic State", add.name, ".txt"))
print(synth.tables)
sink(NULL)



##############################################################
# legislation
##############################################################

dependent = "legislation"
predictors = unique(c(predictors.global, dependent))



dataprep.out <- build.data(dat.synth = dat,
                           predictors = predictors, 
                           dependent = dependent, 
                           treatment.identifier, 
                           controls.identifier,
                           time,
                           time.pretreat)

synth.out <- synth(dataprep.out)

png(filename = "Figure 5. Trajectories of Religious Legislation", width=width, height=height, res=res)
draw.trends(Y0 = (dataprep.out$Y0%*%synth.out$solution.w),
            Y1 = dataprep.out$Y1plot,
            x.range = time,
            ylim = c(0, 45),
            ylab = "Legislation of Religion",
            legend.lab = c("Indonesia","Synthetic Indonesia"),
            time.v = 1998,
            text.x = 1998, text.y = 10, text = "Soeharto's\nDownfall"
)
dev.off()

synth.tables <- synth.tab(
  dataprep.res = dataprep.out,
  synth.res = synth.out)
sink(file = paste0("Synthetic Legislation", add.name, ".txt"))
print(synth.tables)
sink(NULL)



##############################################################
# social
##############################################################

dependent = "social"
predictors = unique(c(predictors.global, dependent))


dataprep.out <- build.data(dat.synth = dat,
                           predictors = predictors, 
                           dependent = dependent, 
                           treatment.identifier, 
                           controls.identifier,
                           time,
                           time.pretreat)

synth.out <- synth(dataprep.out)

png(filename = "Figure 6. Trajectories of Social Discrimination", width=width, height=height, res=res)
draw.trends(Y0 = (dataprep.out$Y0%*%synth.out$solution.w),
            Y1 = dataprep.out$Y1plot,
            x.range = time,
            ylim = c(0, 45),
            ylab = "Social Discrimination of Religious Minorities",
            legend.lab = c("Indonesia","Synthetic Indonesia"),
            time.v = 1998,
            text.x = 1998, text.y = 10, text = "Soeharto's\nDownfall"
)
dev.off()


synth.tables <- synth.tab(
  dataprep.res = dataprep.out,
  synth.res = synth.out)
sink(file = paste0("Synthetic Social", add.name, ".txt"))
print(synth.tables)
sink(NULL)

